//---------------------------------------------------------------------------
/*

  The goal of this example is to show how to use more than one
  subsystem of a single board.

  This example performs a polled mode AI sample, a polled mode DI sample
  and a polled mode AO write.

  For independent operation, allocate three distince SR structures...one for
  each task.

  This example was tested with BCB 5.0 Standard, in Win2K SP4 with
  KPCI-3110 board using 3100-850A03 version of DriverLINX

*/
#include <vcl.h>
#pragma hdrstop

#include "MultiSRmain.h"
#include "drvlinx.h"   /* DriverLINX API */
#include "dlcodes.h"   /* DriverLINX error codes and macros */
// be sure to add c:\drvlinx4\dlapi\drvlnxbc.lib to the project

// DriverLINX variables and function prototypes

DL_SERVICEREQUEST AISR;    // declare the Service Request Structures
DL_SERVICEREQUEST DISR;
DL_SERVICEREQUEST AOSR;
HANDLE hInstance;       // instance handle for the driver

HANDLE hWin;           // handle to this application
int DeviceNum;

// variables used for error handling
char errString[100];
DWORD size=50;
int errNumber;
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TForm1 *Form1;
//---------------------------------------------------------------------------
__fastcall TForm1::TForm1(TComponent* Owner)
        : TForm(Owner)
{
}
//---------------------------------------------------------------------------
void __fastcall TForm1::cmdOpenClick(TObject *Sender)
{
const char driverName1[20] = "kpci3100";  // driver for our single board

/* change the driver name above to match your installed hardware
   "kpci3108" for KPCI-3107 or KPCI-3108
   "kmb1800" for DAS-1802ST and other DAS-1800 Series boards
*/

       // get the handle of this application's window
        hWin = Form1->Handle;
      /* window handle is used for message processing which
         is not really an issue for Polled mode tasks, but for
         IRQ or DMA tasks is vitally important to coordinate
         reading data with data availability.  DriverLINX requires
         a window handle for all tasks even if messages are not in the
         end used by the task
       */

	// call OpenDriverLINX()
	hInstance = (HANDLE)OpenDriverLINX( hWin, driverName1 );
        // use name of your driver to avoid the OpenDriverLINX Dialog box.
        
        Form1->SetFocus();  // set focus back to our application
        // allow Initialization to take place now....
        cmdInit->Enabled = True;
}
//---------------------------------------------------------------------------

void __fastcall TForm1::cmdInitClick(TObject *Sender)
{       // allocate and init members of the SR structures
     	memset( &AISR, 0, sizeof( DL_SERVICEREQUEST ) );
	DL_SetServiceRequestSize( AISR );
        // do the same for the second Service Request, DISR
       	memset( &DISR, 0, sizeof( DL_SERVICEREQUEST ) );
	DL_SetServiceRequestSize( DISR );
        // same for the third SR
        memset( &AOSR, 0, sizeof( DL_SERVICEREQUEST ) );
	DL_SetServiceRequestSize( AOSR );

        DeviceNum = StrToInt(txtDevNum->Text); // read in the device number from form
        // using one SR, init the device which will init all subsystems
	AISR.hWnd = hWin;
        AISR.operation = INITIALIZE;  // carry out an initialization operation
	AISR.device = DeviceNum;  // the device number assigned in DLinx Config Panel
	AISR.subsystem = DEVICE;
	AISR.mode = OTHER;
    	// call DriverLINX() to execute the task specified by the
        // members of the SR structure
	if( DriverLINX( &AISR ) != NoErr )
         {// respond to error
           errNumber = getErrCode(AISR.result);
        // errNumber corresponds to DriverLINX documentation
        ReturnMessageString(AISR.hWnd,AISR.result,errString,size);
        // errString corresponds to DriverLINX documentation
        lblAIStatus->Caption = errString;
        // don't allow use of the subsystems...
        cmdReadAI->Enabled = False;
        cmdReadDI->Enabled = False;
        cmdAOWrite->Enabled = False;
          }
        else
        {// all is well, allow use of the subsystems...
        cmdReadAI->Enabled =True;
        cmdReadDI->Enabled =True;
        cmdAOWrite->Enabled = True;
        }

        /*
           The code below sets up most of the SR members necessary for
           a polled mode AO task.  In the click event for the cmdAOWrite
           button, the value to be written is set and then the DriverLINX()
           function is called to carry out the task, e.g., write the value.
        */
        AOSR.hWnd = hWin;
	AOSR.device = DeviceNum;
	AOSR.subsystem = AO; // use the Analog Output subsystem of the board
	AOSR.mode = POLLED;
	AOSR.operation = START;
 	AOSR.channels.nChannels = 1;  // a single channel
	AOSR.channels.chanGain[ 0 ].channel = 0;  // it is channel 0
        AOSR.channels.chanGain[ 0 ].gainOrRange = Gain2Code(0,AO,-1.0);
        AOSR.taskFlags = NO_SERVICESTART | NO_SERVICEDONE;
        // above line turns off the sending of Service Start and Done messages

}
//---------------------------------------------------------------------------


void __fastcall TForm1::cmdReadDIClick(TObject *Sender)
{
        DISR.hWnd = hWin;
	DISR.device = DeviceNum;
	DISR.subsystem = DI; // use the Digital Input subsystem of the board
	DISR.mode = POLLED;
	DISR.operation = START;
 	DISR.channels.nChannels = 1;  // a single channel will be sampled
	DISR.channels.chanGain[ 0 ].channel = 0;  // it is channel 0
        DISR.taskFlags = NO_SERVICESTART | NO_SERVICEDONE;

	// call DriverLINX()
	if( DriverLINX( &DISR ) != NoErr )
        {
	errNumber = getErrCode(DISR.result);
        // errNumber corresponds to DriverLINX documentation
        ReturnMessageString(DISR.hWnd,DISR.result,errString,size);
        // errString corresponds to DriverLINX documentation
        lblDIStatus->Caption = errString;
        lblDIValue->Caption = "";
	}
        else
        {
         lblDIValue->Caption = DISR.status.u.ioValue;
         lblDIStatus->Caption = "No Error";
        }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::cmdReadAIClick(TObject *Sender)
{
        AISR.hWnd = hWin;
	AISR.device = DeviceNum;
	AISR.subsystem = AI; // use the Analog Input subsystem of the board
	AISR.mode = POLLED;
	AISR.operation = START;
 	AISR.channels.nChannels = 1;  // a single channel
	AISR.channels.chanGain[ 0 ].channel = 0;  // it is channel 0
        AISR.channels.chanGain[ 0 ].gainOrRange = Gain2Code(0,AI,-1.0);
        // the -1 means gain of 1; negative sign means bipolar range for
        // an overall measurement range of -10 to +10 VDC
        AISR.taskFlags = NO_SERVICESTART | NO_SERVICEDONE;

	// call DriverLINX()
	if( DriverLINX( &AISR ) != NoErr )
        {
         errNumber = getErrCode(AISR.result);
        // errNumber corresponds to DriverLINX documentation
        ReturnMessageString(AISR.hWnd,AISR.result,errString,size);
        // errString corresponds to DriverLINX documentation
        lblAIStatus->Caption = errString;
        lblAIValue->Caption = "";
	}
        else
        {
        float volts;
        int result;
        result=Code2Volts(DeviceNum,AI,AISR.status.u.ioValue,&volts); //Convert to volts
        char str[25];
        gcvt(volts, 4, str); // format volts
        lblAIValue->Caption = str;
        lblAIStatus->Caption = "No Error";
        }
}
//---------------------------------------------------------------------------


void __fastcall TForm1::cleanUpDriverLINX(TObject *Sender)
{
  // close the driver, if it was ever opened
        if (hInstance != NULL)
        {
        CloseDriverLINX(hInstance); //close the DriverLINX driver
	hInstance=NULL; //for safety, make sure hInstance isn't pointing to anything
        }
	delete(&AISR); //de-allocate the memory used by the service requests
     	delete(&DISR);
        delete(&AOSR);
}
//---------------------------------------------------------------------------


void __fastcall TForm1::cmdAOWriteClick(TObject *Sender)
{
        float volts;
        volts = StrToFloat(txtAOValue->Text);

        //Convert volts to D/A units used by the DAC chip
        AOSR.status.u.ioValue=Volts2Code(AOSR.device,AO,volts);
        /*
           Volts2Code function assumes the board is using the default
           range which is typically -10 to +10 VDC.  At other ranges, this
           function will not compute the correct values.  Need to use a CONVERT
           operation for correct computation at all the ranges/gains.
        */
        if( DriverLINX( &AOSR ) != NoErr )
        {
         errNumber = getErrCode(AOSR.result);
        // errNumber corresponds to DriverLINX documentation
        ReturnMessageString(AOSR.hWnd,AOSR.result,errString,size);
        // errString corresponds to DriverLINX documentation
        lblAOStatus->Caption = errString;
        }
        else
        {
         //lblAOStatus->Caption = StrCat("Written ",txtAOValue->Text.c_str());
         lblAOStatus->Caption = "AO Written with no error";
        }
}
//---------------------------------------------------------------------------


